# ============================ Treap ==============================

from typing import Tuple, NamedTuple
import random, sys

MAXPRI = 2**20
random.seed(47)

class Treap(NamedTuple):
    key: int  # values are sorted from left to right by key; !!! must be an integer
    prio: int
    my_weight: int  # total weight of subtree incl myself
    weight: int  # total weight of subtree incl myself
    key_sum: int  # total sum of keys of subtree incl myself
    left: "Treap"
    right: "Treap"


def new_treap(
    key: int, prio: int, my_weight: int, left: Treap = None, right: Treap = None
) -> Treap:
    weight = my_weight + (bool(left) and left.weight) + (bool(right) and right.weight)
    key_sum = (
        my_weight * key
        + (bool(left) and left.key_sum)
        + (bool(right) and right.key_sum)
    )
    return Treap(key, prio, my_weight, weight, key_sum, left, right)


def split(T: Treap, key: int) -> Tuple[Treap, Treap]:
    if T is None:
        return None, None
    if T.key <= key:
        L, R = split(T.right, key)
        return new_treap(T.key, T.prio, T.my_weight, T.left, L), R
    else:
        L, R = split(T.left, key)
        return L, new_treap(T.key, T.prio, T.my_weight, R, T.right)


def merge(T1: Treap, T2: Treap) -> Treap:
    if T2 is None:
        return T1
    if T1 is None:
        return T2
    if T1.prio > T2.prio:
        return new_treap(T1.key, T1.prio, T1.my_weight, T1.left, merge(T1.right, T2))
    else:
        return new_treap(T2.key, T2.prio, T2.my_weight, merge(T1, T2.left), T2.right)
class HalfSortedList:
    def __init__(self):
        self.root = None

    def add(self, value: int, weight: int = 1):
        T1, T2 = split(self.root, value)
        T1, k = split(T1, value - 1)
        if k:
            assert k.key == value, "bug in code"
            weight += k.my_weight
            prio = k.prio
        else:
            prio = random.randint(0, MAXPRI)
        self.root = merge(T1, merge(new_treap(value, prio, weight), T2))

    def remove(self, value: int, weight: int = 1):
        T1, T2 = split(self.root, value)
        T1, k = split(T1, value - 1)
        assert (
            k.key == value and k.my_weight >= weight
        ), "you can only remove existing elements"
        if k.weight == weight:
            self.root = merge(T1, T2)
        else:
            self.root = merge(
                T1, merge(new_treap(value, k.prio, k.my_weight - weight), T2)
            )

    def sum_smallest_k(self, k) -> int:
        T = self.root
        skipped_sum = 0
        while T:
            left_weight = bool(T.left) and T.left.weight
            left_sum = bool(T.left) and T.left.key_sum
            if k < left_weight:
                T = T.left
            elif k <= left_weight + T.my_weight:
                return skipped_sum + left_sum + T.key * (k - left_weight)
            else:
                k -= left_weight + T.my_weight
                skipped_sum += left_sum + T.key * T.my_weight
                T = T.right
        raise KeyError("something weird happened")


class SortedList:
    def __init__(self, k):
        self.negatives = HalfSortedList()
        self.positives = HalfSortedList()
        self.sum = 0
        self.k = k

    def add(self, value: int, weight: int = 1):
        self.sum += value * weight
        if self.k > 0 and value < 0:
            self.negatives.add(value, weight)
        if self.k > 0 and value > 0:
            self.positives.add(-value, weight)

    def remove(self, value: int, weight: int = 1):
        self.sum -= value * weight
        if self.k > 0 and value < 0:
            self.negatives.remove(value, weight)
        if self.k > 0 and value > 0:
            self.positives.remove(-value, weight)

    def sum_smallest_k_negatives(self) -> int:
        if self.k == 0 or self.negatives.root is None:
            return 0
        if self.k >= self.negatives.root.weight:
            return self.negatives.root.key_sum
        return self.negatives.sum_smallest_k(self.k)

    def sum_largest_k_positives(self) -> int:
        if self.k == 0 or self.positives.root is None:
            return 0
        if self.k >= self.positives.root.weight:
            return -self.positives.root.key_sum
        return -self.positives.sum_smallest_k(self.k)


# ============================ The actual solution ==============================

N, P, K = [int(x) for x in sys.stdin.readline().split()]
D = []
for n in range(N):
    smer, vzd = sys.stdin.readline().split()
    D.append(int(vzd) if smer == "V" else -int(vzd))

if P == 0:
    print(0)
    quit(0)

S = SortedList(K)
best = 0

for i in range(N):
    S.add(D[i])
    if i >= P:
        S.remove(D[i - P])
    if i >= P - 1:
        best = max(best, S.sum - 2 * S.sum_smallest_k_negatives())
        best = max(best, -S.sum + 2 * S.sum_largest_k_positives())

print(best)
